#!/bin/bash

chmod +x .env.sh

#variables

source .env.sh

####################################################################################
#checking if APT-GET or YUM is supported
checkIfAptgetOrYum(){
	if command -v apt-get >/dev/null; then
		echo "apt-get"
	elif command -v yum >/dev/null; then
		echo "yum"
	else
		echo "miniOrange: apt-get or yum not found"
	fi
}

####################################################################################

#checking if JAVA is installed
checkIfJavaIsInstalled(){
	x=$(java -version 2>&1 | sed -n ';s/.* version "\(.*\)\.\(.*\)\..*".*/\1\2/p;')
	isJavaPresent=$x
	if [ -z "$isJavaPresent" ]
	then
		echo "miniOrange: Java installation is not required. Proceeding..."
	else
		echo "$isJavaPresent"
	fi
}

####################################################################################

#installing miniOrange onpremise idp
installmoIDP(){
	echo "miniOrange: Downloading miniOrange Server"
	cd $moRootFolderName
	wget -4 "$moIDPDownloadLink"
	unzip "mo-idp-server-*"
	rm mo-idp-server-*.zip

	# === Step 2: Assign executable permissions ===
	chmod +x "$moIDPDownloadLink"/bin/*.sh

	# === Step 3: Create Environment Files ===
	echo "Creating /etc/mo-idp-server.env..."
    cat <<EOF | tee /etc/mo-idp-server.env
MO_IDP_FOLDER=/opt/mo-idp-server-4.13.0
MO_CONFIG_PATH=/opt/mo-idp-server-4.13.0/services/config
ROOT_LOG_LEVEL=INFO
MO_LOGS_PATH=/opt/mo-idp-server-4.13.0/services/logs
MO_LOG_LEVEL=DEBUG
EOF

	set -a $miniOrangeEnvFile && source $miniOrangeEnvFile
	set +a $miniOrangeEnvFile
	echo "Environment file /etc/mo-idp-server.env created successfully..."

	systemctl enable $moIdpFolderName/services/linux/mo-idp-miniorange.service
	systemctl daemon-reload
}

####################################################################################

#openTomcat
openTomcat(){
	chmod +x $moIdpFolderName/bin/catalina.sh
	systemctl start mo-idp-miniorange.service
	#exit;
}

####################################################################################

#installing JAVA8
installJava(){
	echo "miniOrange: Would you like to install Java8? ['y' for YES, 'n' for NO, default is YES]"
	read input
	if [ "$input" = "n" ]; then
		echo 'miniOrange: Not installing Java!!!'
		echo "miniOrange: Proceeding with miniOrange Onpremise IDP installation."
		installmoIDP
	else
		if [ "$package" = "apt-get" ]; then
			sudo apt update && sudo apt install openjdk-8-jdk openjdk-8-jre
			sudo update-java-alternatives --set java-1.8.0-openjdk-amd64
		elif [ "$package" = "yum" ]; then
			if checkJdkPackageInYum "java-1.8.0-openjdk"; then
				echo "OpenJDK is available. Installing OpenJDK..."
				sudo yum install -y java-1.8.0-openjdk
			elif checkJdkPackageInYum "java-1.8.0-amazon-corretto"; then
				echo "Correto is available. Installing Correto..."
				yum install -y java-1.8.0-amazon-corretto
			else
				echo "miniOrange: Java package not found in yum repository! Terminating!"
				exit
			fi
		else
			echo "miniOrange: Java package not found! Terminating!"
			exit
		fi
		echo "miniOrange: Java installation completed!"
		installmoIDP
	fi
}

####################################################################################

checkJdkPackageInYum() {
    yum list available "$1" &>/dev/null
    return $?
}

####################################################################################

beginInstallation(){
	echo "miniOrange: Welcome to the installation of miniOrange Onpremise IDP."
	echo "miniOrange: Checking if required Java version is installed."
	javaVersion=$(checkIfJavaIsInstalled)
	if [ "$javaVersion" != "$javaRequiredVersion" ]; then
		echo "miniOrange: Java 8 Not Found"
		installJava
	else
		echo "miniOrange: Java $javaVersion installed"
		echo "miniOrange: Skipping Java installation"
		installmoIDP
	fi
}

####################################################################################

setEnvironmentVariables(){
	echo "Enabling all environment variables."
	echo "Create env file if it does not already exist"
	touch $miniOrangeEnvFile
	echo "Setting environment variables in $miniOrangeEnvFile."
	if ! grep -q "^MO_IDP_FOLDER=" $miniOrangeEnvFile; then
		echo "MO_IDP_FOLDER=$moIdpFolderName" >> $miniOrangeEnvFile
	fi
	if ! grep -q "^MO_CONFIG_PATH=" $miniOrangeEnvFile; then
		echo "MO_CONFIG_PATH=$moIdpFolderName/services/config" >> $miniOrangeEnvFile
	fi
	if ! grep -q "^ROOT_LOG_LEVEL=" $miniOrangeEnvFile; then
		echo "ROOT_LOG_LEVEL=$rootLogLevel" >> $miniOrangeEnvFile
	fi
	if ! grep -q "^MO_LOGS_PATH=" $miniOrangeEnvFile; then
		echo "MO_LOGS_PATH=$moIdpFolderName"/services/logs >> $miniOrangeEnvFile
	fi
	if ! grep -q "^MO_LOG_LEVEL=" $miniOrangeEnvFile; then
		echo "MO_LOG_LEVEL=$miniOrangeLogLevel" >> $miniOrangeEnvFile
	fi
}

####################################################################################

checkIfNginxIsInstalled() {
    if ! command -v nginx &> /dev/null; then
        echo "Nginx not installed. Need to install."
        setupNginx
    else
        echo "miniOrange: Nginx installation is not required. Proceeding..."
    fi
}

####################################################################################

setupNginx(){
	echo "nginx Setup required"
	package=$(checkIfAptgetOrYum)
	if [ "$package" = "apt-get" ]; then
		sudo apt update && sudo apt install nginx
	elif [ "$package" = "yum" ]; then
		yum -y update
		yum -y install nginx
	else
		echo "miniOrange: Some package are not found! Terminating!"
		exit
	fi
	sudo systemctl enable nginx
	echo "miniOrange: Nginx installation completed!"
}

####################################################################################

copyNginxConfig(){
	echo "Copying Nginx config files"
	cp -r $moIdpFolderName/services/linux/nginx.conf $nginxRoot/nginx.conf
	echo "Copying SSl certificates"
	if [ ! -d "/etc/nginx/conf" ]; then
		echo "Nginx conf folder does not exist. Creating"
		mkdir /etc/nginx/conf
	else
		echo "Folder exists."
	fi
	
	cp -r $moIdpFolderName/services/certs/self-signed* $nginxRoot/conf
}

####################################################################################

restartNginx() {
	echo "Restarting nginx"
	systemctl restart nginx
	echo "Nginx restarted"
}

####################################################################################

enableServices() {
	echo "Enable Services"
	systemctl enable $moIdpFolderName/services/linux/mo-idp-apps.service
	systemctl enable $moIdpFolderName/services/linux/mo-idp-auditing.service
	systemctl enable $moIdpFolderName/services/linux/mo-idp-configserver.service
	systemctl enable $moIdpFolderName/services/linux/mo-idp-eurekaserver.service
	systemctl enable $moIdpFolderName/services/linux/mo-idp-gatekeeper.service
	systemctl enable $moIdpFolderName/services/linux/mo-idp-miniorange.service
	systemctl enable $moIdpFolderName/services/linux/mo-idp-provisioning.service
	systemctl enable $moIdpFolderName/services/linux/mo-idp-directory.service
	systemctl enable $moIdpFolderName/services/linux/mo-idp-scheduler.service
	systemctl enable $moIdpFolderName/services/linux/mo-idp-siem.service
	systemctl enable $moIdpFolderName/services/linux/mo-idp-workflow.service
	systemctl daemon-reload
}

####################################################################################
startIdp() {
	systemctl start mo-idp-miniorange.service
}

####################################################################################

checkIfRedisIsInstalled() {
    if ! command -v redis-cli &> /dev/null; then
        echo "Redis not installed. Need to install."
        installRedis
    else
        echo "miniOrange: Redis installation is not required. Proceeding..."
    fi
}

####################################################################################

installRedis() {
	package=$(checkIfAptgetOrYum)
	if [ "$package" = "apt-get" ]; then
		sudo apt update && sudo apt install redis-server
		sudo systemctl enable redis-server
		sudo systemctl start redis-server
	elif [ "$package" = "yum" ]; then
		yum -y update && yum install redis
		if which redis &>/dev/null; then
			echo "Redis is installed"
			sudo systemctl enable redis
			sudo systemctl start redis
		else
			echo "Redis not installed. Trying Redis6"
			yum -y install redis6
			sudo systemctl enable redis6
			sudo systemctl start redis6
		fi
		
	else
		echo "miniOrange: Some package are not found! Terminating!"
		exit
	fi
	echo "Redis installed"
}

####################################################################################

checkIfWgetisInstalled() {
	if ! command -v wget &> /dev/null; then
        echo "wget not installed. Need to install."
        installWget
    else
        echo "miniOrange: wget installation is not required. Proceeding..."
    fi
}

####################################################################################

installWget() {
	package=$(checkIfAptgetOrYum)
	if [ "$package" = "apt-get" ]; then
		sudo apt update && sudo apt install wget
	elif [ "$package" = "yum" ]; then
		yum -y update && yum install wget
	else
		echo "miniOrange: Some package are not found! Terminating!"
		exit
	fi
}

####################################################################################

checkIfUnzipisInstalled() {
	if ! command -v unzip &> /dev/null; then
        echo "Unzip not installed. Need to install."
        installUnzip
    else
        echo "miniOrange: Unzip installation is not required. Proceeding..."
    fi
}

####################################################################################

installUnzip() {
	package=$(checkIfAptgetOrYum)
	if [ "$package" = "apt-get" ]; then
		sudo apt update && sudo apt install unzip
	elif [ "$package" = "yum" ]; then
		yum -y update && yum install unzip
	else
		echo "miniOrange: Some package are not found! Terminating!"
		exit
	fi
	echo "Unzip installed"
}

####################################################################################

seLinuxCheck() {
	if command -v sestatus &>/dev/null; then
		echo "SELinux is installed. Enable httpd_can_network_relay"
		setsebool -P httpd_can_network_relay 1
		setsebool -P httpd_can_network_connect 1
	else
		echo "SELinux is NOT installed."
	fi
}

addHealthCheckForGatekeeper() {
	echo "Settting healthcheck for Gatekeeper..."
	cp -r $moIdpFolderName/services/linux/healthcheck.sh /etc/healthcheck.sh
	chmod +x /etc/healthcheck.sh
}


####################################################################################

#Installation with elevated privileges
if [[ $EUID -ne 0 ]]; then
    echo "Need to run script with sudo or elevated privileges..."
    exec sudo bash "$0" "$@"
else
	echo "Running with elevated privileges"
fi

#Set environment variables before initializing anything
setEnvironmentVariables
seLinuxCheck
checkIfNginxIsInstalled
checkIfRedisIsInstalled
checkIfWgetisInstalled
checkIfUnzipisInstalled
beginInstallation
copyNginxConfig
restartNginx
enableServices
addHealthCheckForGatekeeper
startIdp
echo "Setup is now complete. Navigate to https://<IP/DNS/localhost>"